class ImageGallery {
            constructor() {
                this.images = [
                    {
                        id: 1,
                        src: 'https://picsum.photos/800/600?random=1',
                        thumb: 'https://picsum.photos/400/300?random=1',
                        title: 'Mountain Landscape',
                        description: 'Breathtaking mountain views with misty peaks and serene valleys.',
                        category: 'nature',
                        tags: ['mountains', 'landscape', 'nature']
                    },
                    {
                        id: 2,
                        src: 'https://picsum.photos/800/600?random=2',
                        thumb: 'https://picsum.photos/400/300?random=2',
                        title: 'Modern Architecture',
                        description: 'Contemporary building design with clean lines and geometric patterns.',
                        category: 'architecture',
                        tags: ['building', 'modern', 'design']
                    },
                    {
                        id: 3,
                        src: 'https://picsum.photos/800/600?random=3',
                        thumb: 'https://picsum.photos/400/300?random=3',
                        title: 'Abstract Colors',
                        description: 'Vibrant abstract composition with flowing colors and dynamic forms.',
                        category: 'abstract',
                        tags: ['colors', 'abstract', 'art']
                    },
                    {
                        id: 4,
                        src: 'https://picsum.photos/800/600?random=4',
                        thumb: 'https://picsum.photos/400/300?random=4',
                        title: 'Forest Path',
                        description: 'Peaceful woodland trail surrounded by ancient trees and dappled sunlight.',
                        category: 'nature',
                        tags: ['forest', 'trees', 'path']
                    },
                    {
                        id: 5,
                        src: 'https://picsum.photos/800/600?random=5',
                        thumb: 'https://picsum.photos/400/300?random=5',
                        title: 'City Skyline',
                        description: 'Urban landscape showcasing towering skyscrapers and city lights.',
                        category: 'architecture',
                        tags: ['city', 'skyline', 'urban']
                    },
                    {
                        id: 6,
                        src: 'https://picsum.photos/800/600?random=6',
                        thumb: 'https://picsum.photos/400/300?random=6',
                        title: 'Geometric Patterns',
                        description: 'Intricate geometric design with symmetrical patterns and bold contrasts.',
                        category: 'abstract',
                        tags: ['geometry', 'patterns', 'design']
                    },
                    {
                        id: 7,
                        src: 'https://picsum.photos/800/600?random=7',
                        thumb: 'https://picsum.photos/400/300?random=7',
                        title: 'Ocean Waves',
                        description: 'Powerful ocean waves crashing against rocky shores under dramatic skies.',
                        category: 'nature',
                        tags: ['ocean', 'waves', 'water']
                    },
                    {
                        id: 8,
                        src: 'https://picsum.photos/800/600?random=8',
                        thumb: 'https://picsum.photos/400/300?random=8',
                        title: 'Glass Architecture',
                        description: 'Stunning glass facade reflecting the surrounding environment.',
                        category: 'architecture',
                        tags: ['glass', 'reflection', 'modern']
                    },
                    {
                        id: 9,
                        src: 'https://picsum.photos/800/600?random=9',
                        thumb: 'https://picsum.photos/400/300?random=9',
                        title: 'Fluid Motion',
                        description: 'Dynamic abstract representation of movement and energy in motion.',
                        category: 'abstract',
                        tags: ['motion', 'fluid', 'energy']
                    }
                ];
                
                this.currentIndex = 0;
                this.currentFilter = 'all';
                this.currentView = 'grid';
                this.zoomLevel = 1;
                this.filteredImages = [...this.images];
                
                this.init();
            }

            init() {
                this.renderGallery();
                this.setupEventListeners();
                this.setupKeyboardNavigation();
            }

            renderGallery() {
                const gallery = document.getElementById('gallery');
                gallery.innerHTML = '';

                this.filteredImages.forEach((image, index) => {
                    const item = document.createElement('div');
                    item.className = 'gallery-item';
                    item.dataset.category = image.category;
                    item.dataset.index = index;

                    item.innerHTML = `
                        <img class="gallery-image" src="${image.thumb}" alt="${image.title}" loading="lazy">
                        <div class="gallery-overlay">
                            <div class="overlay-icon">🔍</div>
                        </div>
                        <div class="gallery-info">
                            <h3>${image.title}</h3>
                            <p>${image.description}</p>
                            <div class="gallery-tags">
                                ${image.tags.map(tag => `<span class="gallery-tag">${tag}</span>`).join('')}
                            </div>
                        </div>
                    `;

                    item.addEventListener('click', () => this.openLightbox(index));
                    gallery.appendChild(item);
                });
            }

            setupEventListeners() {
                // Filter buttons
                document.querySelectorAll('.filter-btn').forEach(btn => {
                    btn.addEventListener('click', (e) => {
                        document.querySelectorAll('.filter-btn').forEach(b => b.classList.remove('active'));
                        e.target.classList.add('active');
                        this.filterImages(e.target.dataset.filter);
                    });
                });

                // View buttons
                document.querySelectorAll('.view-btn').forEach(btn => {
                    btn.addEventListener('click', (e) => {
                        document.querySelectorAll('.view-btn').forEach(b => b.classList.remove('active'));
                        e.target.classList.add('active');
                        this.changeView(e.target.dataset.view);
                    });
                });

                // Search box
                document.querySelector('.search-box').addEventListener('input', (e) => {
                    this.searchImages(e.target.value);
                });

                // Lightbox controls
                document.getElementById('closeBtn').addEventListener('click', () => this.closeLightbox());
                document.getElementById('prevBtn').addEventListener('click', () => this.previousImage());
                document.getElementById('nextBtn').addEventListener('click', () => this.nextImage());
                
                // Zoom controls
                document.getElementById('zoomIn').addEventListener('click', () => this.zoomImage(1.2));
                document.getElementById('zoomOut').addEventListener('click', () => this.zoomImage(0.8));
                document.getElementById('zoomReset').addEventListener('click', () => this.resetZoom());

                // Click outside lightbox to close
                document.getElementById('lightbox').addEventListener('click', (e) => {
                    if (e.target.id === 'lightbox') {
                        this.closeLightbox();
                    }
                });
            }

            setupKeyboardNavigation() {
                document.addEventListener('keydown', (e) => {
                    if (document.getElementById('lightbox').classList.contains('active')) {
                        switch(e.key) {
                            case 'Escape':
                                this.closeLightbox();
                                break;
                            case 'ArrowLeft':
                                this.previousImage();
                                break;
                            case 'ArrowRight':
                                this.nextImage();
                                break;
                            case '+':
                            case '=':
                                this.zoomImage(1.2);
                                break;
                            case '-':
                                this.zoomImage(0.8);
                                break;
                            case '0':
                                this.resetZoom();
                                break;
                        }
                    }
                });
            }

            filterImages(category) {
                this.currentFilter = category;
                this.applyFilters();
            }

            searchImages(query) {
                this.searchQuery = query.toLowerCase();
                this.applyFilters();
            }

            applyFilters() {
                let filtered = [...this.images];

                // Apply category filter
                if (this.currentFilter !== 'all') {
                    filtered = filtered.filter(img => img.category === this.currentFilter);
                }

                // Apply search filter
                if (this.searchQuery) {
                    filtered = filtered.filter(img => 
                        img.title.toLowerCase().includes(this.searchQuery) ||
                        img.description.toLowerCase().includes(this.searchQuery) ||
                        img.tags.some(tag => tag.toLowerCase().includes(this.searchQuery))
                    );
                }

                this.filteredImages = filtered;
                this.renderGallery();
            }

            changeView(view) {
                const gallery = document.getElementById('gallery');
                this.currentView = view;
                
                if (view === 'list') {
                    gallery.classList.add('list-view');
                } else {
                    gallery.classList.remove('list-view');
                }
            }

            openLightbox(index) {
                this.currentIndex = index;
                const image = this.filteredImages[index];
                const lightbox = document.getElementById('lightbox');
                const lightboxImage = document.getElementById('lightboxImage');
                const lightboxTitle = document.getElementById('lightboxTitle');
                const lightboxDescription = document.getElementById('lightboxDescription');

                lightboxImage.src = image.src;
                lightboxTitle.textContent = image.title;
                lightboxDescription.textContent = image.description;
                
                lightbox.classList.add('active');
                document.body.style.overflow = 'hidden';
                this.resetZoom();
            }

            closeLightbox() {
                document.getElementById('lightbox').classList.remove('active');
                document.body.style.overflow = 'auto';
            }

            previousImage() {
                this.currentIndex = (this.currentIndex - 1 + this.filteredImages.length) % this.filteredImages.length;
                this.updateLightboxImage();
            }

            nextImage() {
                this.currentIndex = (this.currentIndex + 1) % this.filteredImages.length;
                this.updateLightboxImage();
            }

            updateLightboxImage() {
                const image = this.filteredImages[this.currentIndex];
                const lightboxImage = document.getElementById('lightboxImage');
                const lightboxTitle = document.getElementById('lightboxTitle');
                const lightboxDescription = document.getElementById('lightboxDescription');

                lightboxImage.src = image.src;
                lightboxTitle.textContent = image.title;
                lightboxDescription.textContent = image.description;
                this.resetZoom();
            }

            zoomImage(factor) {
                this.zoomLevel *= factor;
                this.zoomLevel = Math.max(0.5, Math.min(3, this.zoomLevel));
                document.getElementById('lightboxImage').style.transform = `scale(${this.zoomLevel})`;
            }

            resetZoom() {
                this.zoomLevel = 1;
                document.getElementById('lightboxImage').style.transform = 'scale(1)';
            }
        }

        // Initialize gallery when page loads
        document.addEventListener('DOMContentLoaded', () => {
            new ImageGallery();
        });